/*
* Copyright 2019-2025 Amazon.com, Inc. or its affiliates. All Rights Reserved.
*
* SPDX-License-Identifier:  GPL-2.0+
*/

#include <common.h>
#include <ring_platform_features.h>
#include "asm/arch/mach/platform.h"
#include <drvAESDMA.h>
#include <bootm.h>
#include <malloc.h>
#include <nand.h>
#include <fb_otp.h>
#include <spinand.h>
#include <drvSPINAND.h>
#include <unlocker.h>
#include <cli.h>
#include <fastboot.h>
#include <ubi_uboot.h>
#include <environment.h>
#ifdef COCOA_INCLUDE_KEYS_IN_IMAGE
#include <keys_header.h>
#endif
#include <ring_boot_config.h>

#ifdef DEBUG
#define debugf(fmt, args...) do { printf("%s(): ", __func__); printf(fmt, ##args); } while (0)
#else
#define debugf(fmt, args...)
#endif

#define STR_HELPER(x) #x
#define STR(x) STR_HELPER(x)

#ifndef COCOA_KERNEL_SIZE
#error Need to define kernel image size in COCOA_KERNEL_SIZE
#endif

#ifndef COCOA_UBI_MTD
#error Need to define ubi mtd number in COCOA_UBI_MTD
#endif

#ifndef COCOA_RECOVERY_KERNEL_PART_NAME
#error Need to define recovery kernel image partition name
#endif

#ifndef COCOA_RECOVERY_BACKUP_KERNEL_PART_NAME
#error Need to define recovery backup kernel image partition name
#endif

#ifndef COCOA_RECOVERY_KERNEL_LOAD_ADDRESS
#error Need to define the recovery kernel load address
#endif

#ifndef COCOA_USERDATA_SIZE
#error Need to specify userdata size in bytes
#endif

#ifndef COCOA_DATA_PARTITION
#error Need to specify data partition name
#endif

#ifndef COCOA_RECOVERY_SIZE
#error Need to specify recovery size in bytes
#endif

#ifndef COCOA_KERNEL_DISABLE_RTOS_COMMS
#error Need to specify kernel cmdline parameters to disable RTOS comms
#endif

#ifndef COCOA_KERNEL_DISABLE_RED_COMMS
#error Need to specify kernel cmdline parameters to disable Red comms
#endif


#define COCOA_ENV_NAME "ENV"
#define COCOA_UBI_NAME "UBI"
#define COCOA_RECOVERY_PARTITION "recovery"
#define COCOA_SHADOW_RECOVERY_PARTITION "shadow"
#define COCOA_BOOT_CONFIG_PARTITION "bootconfig"
#define COCOA_NEXT_STAGE_SIG_SIZE 256
#define COCOA_KERNEL_SIG_SIZE 256

#define NOCONSOLE_ARG     "console= "
#define NOCONSOLE_ARG_LEN (sizeof(NOCONSOLE_ARG) - 1)

#define UNLOCKED_KERNEL_ROOT "ringboot.unlocked_kernel="
#define SERIALNO_ROOT "ringboot.serialno="
#define BOARDID_ROOT " ringboot.boardid="

#define LOCKED_KERNEL_ARG       UNLOCKED_KERNEL_ROOT "false "
#define LOCKED_KERNEL_ARG_LEN   (sizeof(LOCKED_KERNEL_ARG) - 1)
#define UNLOCKED_KERNEL_ARG     UNLOCKED_KERNEL_ROOT "true "
#define UNLOCKED_KERNEL_ARG_LEN (sizeof(UNLOCKED_KERNEL_ARG) - 1)

#define BACKUP_KERNEL_STR "ringboot.backup_kernel"
#define BACKUP_KERNEL_STR_LEN (sizeof(BACKUP_KERNEL_STR) - 1)

#define USE_SHADOW_ARG "ringboot.rootfs_name=" COCOA_SHADOW_RECOVERY_PARTITION
#define USE_SHADOW_ARG_LEN  (sizeof(USE_SHADOW_ARG) - 1)

// Maximum number of repeated recovery boots allowed
#define MAX_RECOVERY_MODE_BOOT_COUNT 5

// Maximum number of repeated normal boots without confirmation of success allowed
#define MAX_NORMAL_MODE_BOOT_COUNT 8

#ifdef MTDPARTS_ON_KERNEL_CMDLINE
#define MTDPARTS_MAXLEN             512
#endif

// Length of array used to construct sensor string of form sensor1+sensor2
#define SENSOR_STR_LEN              32

#ifdef COCOA_INCLUDE_KEYS_IN_IMAGE
// Next stage public key
static u8 kernel_prod_key[] = { PROD_KERNEL_PUB };
static u8 kernel_dev_key[]  = { DEV_KERNEL_PUB };
static u8 ns_prod_key[] = { COCOA_NEXT_STAGE_PROD_KEY };
static u8 ns_dev_key[]  = { COCOA_NEXT_STAGE_DEV_KEY };

// Device unlock public keys
static u8 dev_unlock_key[]    = { PROD_UNLOCK_DEV_PUB };
static u8 vendor_unlock_key[] = { PROD_UNLOCK_VENDOR_PUB };
#endif

/* Exclude offsets outside the flashed area and
   avoid fetching the pub key that contains 0xFF only
*/
static const int PROD_PAGE_MAX_OFFSETS[OTP_VERSION_MAX] = {MFG_UNLOCK_KEY_OFFSET, IPL_KEY_OFFSET};
static const int DEV_PAGE_MAX_OFFSETS[OTP_VERSION_MAX] = {RTOS_KEY_OFFSET, RTOS_KEY_OFFSET};

/* This is intended as a failsafe against programming errors which inadvertently
 * skip authentication and hence allow boot.  Unless runAuthenticate has returned
 * successfully, this value should not be set in the "has_authenticated"
 * variable
 */
#define HAS_AUTHENTICATED_KEY   0x41757468
#define NO_AUTH_KEY             0x44657641
static uint32_t has_authenticated = 0;

#define PRIV_BOOTARGS_LEN 1024
static char *private_bootargs = NULL;

static char dsn[COCOA_DSN_LEN+1];
static char boardid[COCOA_BOARD_ID_LEN+1];
static char sensorstr[SENSOR_STR_LEN];

static const unsigned int BOARD_ID_PROD_ID_INDEX; // = 0
static const unsigned int BOARD_ID_PROD_ID_LEN      =  4;
static const unsigned int BOARD_ID_2ND_SOURCE_INDEX =  4;
static const unsigned int BOARD_ID_MAJOR_REV_INDEX  =  7;
static const unsigned int BOARD_ID_MINOR_REV_INDEX  =  8;

static const char * const BOARD_ID_PROD_ID_STRS[] = {
	"c0ca",
	"c0cb",
	"c0cc",
	"c0cd",
	"c0ce",
	"c0cf",
	"c0d0",
	"07ca",
	"c2ca",
	"c2cb",
	"c2cc",
	"c0d2"
};

enum RB_PROD_ID_TYPES {
	RB_PRODUCT_ABA,
	RB_PRODUCT_ABC,
	RB_PRODUCT_ABG,
	RB_PRODUCT_ABR,
	RB_PRODUCT_ABP,
	RB_PRODUCT_ABN,
	RB_PRODUCT_ABM,
	RB_PRODUCT_ABO,
	RB_PRODUCT_ABS,
	RB_PRODUCT_ABE,
	RB_PRODUCT_ABF,
	RB_PRODUCT_ABH,
	RB_PRODUCT_UNKNOWN
};

static const char *ps5250_str = "ps5250";
static const char *ps5416_str = "ps5416";
static const char *sp2309_str = "sp2309";
static const char *sns4_str  = "sns4";
static const char *sns3_str = "sns3";
static const char *sns2_str = "sns2";
static const char *os04c10_str = "os04c10";
static const char *sns1_str = "sns1";
static const char *bt656_str = "bt656";

enum RB_MAJOR_REVS {
	RB_MAJOR_REV_UNKNOWN = 0,

	RB_MAJOR_REV_MIN = '0',

	RB_MAJOR_REV_PROTO = RB_MAJOR_REV_MIN,
	RB_MAJOR_REV_HVT,
	RB_MAJOR_REV_EVT,
	RB_MAJOR_REV_DVT,
	RB_MAJOR_REV_PVT,
	RB_MAJOR_REV_MP,

	RB_MAJOR_REV_MAX
};


typedef struct rtk_image_header {
	u32     ivt[8];     // Interrupt Vector Table, 32byte
	u32     marker;     // RTK header, 0x5F4B5452 (“RTK_”)
	u32     size;       // Image size (include header)
	u32     checksum;   // Checksum (exclude header)
	u32     reserved;   // Reserved
} rtk_image_header_t;

void signal_boot_ack_pulse(void);
void configure_boot_ack(void);
uint32_t getotpversion(void);

static void alloc_bootargs(void)
{
	if (!private_bootargs) {
		private_bootargs = malloc(PRIV_BOOTARGS_LEN);
		if (private_bootargs)
			private_bootargs[0] = '\0';
	}
}

char *get_bootargs(void)
{
	alloc_bootargs();
	return private_bootargs;
}

int set_bootargs(const char *value)
{
	alloc_bootargs();
	if (private_bootargs == NULL)
		return -1;
	strlcpy(private_bootargs, value, PRIV_BOOTARGS_LEN);
	return 0;
}

#ifdef MTDPARTS_ON_KERNEL_CMDLINE
// returns the safe number of characters that should be used from mtdparts
// protects against something invalid being added to the kernel command line
static int mtd_parts_length(const char *mtdparts)
{
	const char key[] = "mtdparts=";
	int sz;
	if (strncmp(mtdparts, key, sizeof(key) - 1) != 0) {
		return 0;
	}
	for(sz = sizeof(key) - 1; sz < MTDPARTS_MAXLEN; sz++) {
		if (mtdparts[sz] == 0 || mtdparts[sz] == ' ') {
			return sz;
		}
	}
	return MTDPARTS_MAXLEN;
}
#endif

static int strcat_chk(char *destination, const char *source, size_t copy_len, size_t max_dst_size)
{
	const size_t dst_used = strlen(destination);
	if (dst_used < max_dst_size) {
		const size_t src_len = min(strlen(source), copy_len) + 1; // source length to copy (including terminating '\0')
		size_t num_to_copy = min(src_len, max_dst_size - dst_used);
		strlcpy(destination + dst_used, source, num_to_copy);
	}
	/* return matches strlcat */
	return dst_used + copy_len;
}

/* Forward declarations for functions to recreate partitions */
int ubifs_init(void);
int uboot_ubifs_mount(char *vol_name);
int ubi_remove_vol(char *volume);
int ubi_create_vol(char *volume, int64_t size, int dynamic);
int ubi_create_max_size_vol(char *volume, int dynamic);
int ubi_check_size_enough(char *name, int64_t size);
int ubi_check(char *name);

/* Check the recovery rootfs
 * Returns true if the recovery rootfs appears to be mountable
 */
static bool check_recovery_mountable(void)
{
	if (uboot_ubifs_mount("ubi:" COCOA_RECOVERY_PARTITION) != 0) {
		printf("Recovery rootfs is corrupt\n");
		return false;
	}
	return true;
}

/* Check we can mount shadow recovery rootfs
 * Returns true for a valid mountable partition
 */
static bool check_shadow_recovery(void)
{
	if (uboot_ubifs_mount("ubi:" COCOA_SHADOW_RECOVERY_PARTITION) != 0) {
		// We don't create the partition here since we need the contents to be valid
		return false;
	}
	return true;
}

static void set_ring_recovery_command_line(unlock_state_t unlock_state,
					   uint8_t reboot_reason,
					   bool use_backup_kernel)
{
	const char args[] = STR(COCOA_KERNEL_DISABLE_RTOS_COMMS) " rdinit=/init_recovery ringboot.validated_kernel=true ";
	int err = try_otp_read_all(dsn, boardid);

	bool disable_red_comms = true;
	bool use_shadow_recovery = false;
	switch (reboot_reason) {
		case REBOOT_REASON_RECOVERY_STG1_BOOT:
			disable_red_comms = false;
			use_shadow_recovery = false;
			break;
		case REBOOT_REASON_RECOVERY_STG2_BOOT:
			disable_red_comms = false;
			/* We should only be in this state if shadow partition is valid.
			 * The procedure only works if we can boot to shadow as we need
			 * to update recovery rootfs */
			use_shadow_recovery = true;
			break;
		// Treat all other cases the same
		case REBOOT_REASON_APPLY_OTA:
		default:
			disable_red_comms = true;
			if (!check_recovery_mountable())
			{
				use_shadow_recovery = check_shadow_recovery();
			}
			break;
	}

	strlcpy(get_bootargs(), args, PRIV_BOOTARGS_LEN);
	if (disable_red_comms) {
		strcat_chk(get_bootargs(), STR(COCOA_KERNEL_DISABLE_RED_COMMS), sizeof(STR(COCOA_KERNEL_DISABLE_RED_COMMS)), PRIV_BOOTARGS_LEN);
		strcat_chk(get_bootargs(), " ", 1, PRIV_BOOTARGS_LEN);
	}
	if (unlock_state < UNLOCK_STATE_RESTRICTED_UNLOCK) {
		strcat_chk(get_bootargs(), NOCONSOLE_ARG, NOCONSOLE_ARG_LEN, PRIV_BOOTARGS_LEN);
		strcat_chk(get_bootargs(), LOCKED_KERNEL_ARG, LOCKED_KERNEL_ARG_LEN, PRIV_BOOTARGS_LEN);
	} else {
		strcat_chk(get_bootargs(), UNLOCKED_KERNEL_ARG, UNLOCKED_KERNEL_ARG_LEN, PRIV_BOOTARGS_LEN);
	}
	if (use_shadow_recovery) {
		strcat_chk(get_bootargs(), USE_SHADOW_ARG, USE_SHADOW_ARG_LEN, PRIV_BOOTARGS_LEN);
		strcat_chk(get_bootargs(), " ", 1, PRIV_BOOTARGS_LEN);
	}
	if (err >= ERR_METADATA_READ_OK) {
		strcat_chk(get_bootargs(), SERIALNO_ROOT, sizeof(SERIALNO_ROOT)-1, PRIV_BOOTARGS_LEN);
		strcat_chk(get_bootargs(), dsn, COCOA_DSN_LEN, PRIV_BOOTARGS_LEN);

		strcat_chk(get_bootargs(), BOARDID_ROOT, sizeof(BOARDID_ROOT)-1, PRIV_BOOTARGS_LEN);
		strcat_chk(get_bootargs(), boardid, COCOA_BOARD_ID_LEN, PRIV_BOOTARGS_LEN);
	}

#ifdef MTDPARTS_ON_KERNEL_CMDLINE
	char mtdparts_local[MTDPARTS_MAXLEN];
	char *mtdparts = mtdparts_local;
	if (MDrv_SPINAND_GetMtdParts(mtdparts) != 0) {
		printf("Failed to retrieve mtdparts from PNI/SNI.  Using env\n");
		mtdparts = getenv("mtdparts");
	}
	strcat_chk(get_bootargs(), " ", 1, PRIV_BOOTARGS_LEN);
	strcat_chk(get_bootargs(), mtdparts, mtd_parts_length(mtdparts), PRIV_BOOTARGS_LEN);
#endif

	if (use_backup_kernel) {
		strcat_chk(get_bootargs(), " ", 1, PRIV_BOOTARGS_LEN);
		strcat_chk(get_bootargs(), BACKUP_KERNEL_STR, BACKUP_KERNEL_STR_LEN, PRIV_BOOTARGS_LEN);
	}

	printf("Recovery command line: %s\n", get_bootargs());
}

extern int get_part(const char *partname, int *idx, loff_t *off, loff_t *size,
		loff_t *maxsize);
static int read_nand_partition(const char *part_name, size_t size_to_read,
								u_char *addr, int dev)
{
	nand_info_t *nand = &nand_info[dev];
	int idx;
	loff_t off;
	loff_t part_size;
	loff_t maxsize;
	int err = -1;

	err = get_part(part_name, &idx, &off, &part_size, &maxsize);

	if ((err == 0) && (size_to_read <= part_size)) {
		err = nand_read_skip_bad(nand, off, &size_to_read, NULL,
			part_size, addr);
	}
	return err;
}

int read_otp_page(int page, u_char *addr, size_t size_to_read, int dev)
{
	nand_info_t *nand = &nand_info[dev];
	loff_t off = 0;
	int err = -1;

	size_t pagesize = nand->writesize;
	size_t readsize;

	// Dummy read of a partition - required with WB 2G flash to ensure correct plane setting
	// No check of the return code - we don't care about the actual read result here, only the OTP read
	// See COCOA-13149
	read_nand_partition("IPL0", size_to_read > 128 ? 128 : size_to_read, addr, 0);

	loff_t base = MDrv_SPINAND_GetOTPPageOffset(false, false);
	if (MDrv_SPINAND_OTPInNormalBlock() && nand_block_isbad(nand, base * pagesize)) {
		base = MDrv_SPINAND_GetOTPPageOffset(true, false);
	}

	const size_t nCopies = MDrv_SPINAND_GetOTPMaxBackup();

	for (size_t i = 0; i < nCopies; i++) {
		off = (base + page + (i * OTP_GROUP_SIZE)) * pagesize;
		readsize = size_to_read;

		if (size_to_read <= pagesize) {
			MDrv_SPINAND_EnableOtp(true);
			err = nand_read_skip_bad(nand, off, &readsize, NULL, pagesize, addr);
			MDrv_SPINAND_EnableOtp(false);
		}

		if (!err) {
			break;
		}
		printf("otp read err = %d\n", err);
	}

	return err;
}

#define authfail(err) \
	do { \
		printf("Authentication failed! (err=%d)\n", err); \
		goto exit_authfail; \
	} while(0);

static int authenticate_image(u32 image_address, u32 image_size,
                              u32 key_offset, unlock_state_t unlock_state,
                              bool use_backup)
{
	int err;
	u8* ringkey = (u8*)KEY_CUST_LOAD_ADDRESS;

	err = load_ring_public_key(ringkey, PROD_KEY_PAGE, key_offset,
	                           RSA_KEYSIZE);
	if (err)
		authfail(err);

	if (runAuthenticate((U32)image_address, image_size, (U32*)ringkey)) {
		has_authenticated = HAS_AUTHENTICATED_KEY;
	} else {
		if (unlock_state < UNLOCK_STATE_RESTRICTED_UNLOCK) {
			err = -1;
			authfail(err);
		} else {
			printf("Attempt authentication with dev keys\n");
			err = load_ring_public_key(ringkey, DEV_KEY_PAGE,
			                           key_offset, RSA_KEYSIZE);
			if (err) {
				authfail(err);
			} else {
				if (runAuthenticate((U32)image_address,
				                     image_size,
				                     (U32*)ringkey)) {
					has_authenticated = HAS_AUTHENTICATED_KEY;
				} else if (unlock_state < UNLOCK_STATE_NON_PROD) {
					err = -2;
					authfail(err);
				} else {
					has_authenticated = NO_AUTH_KEY;
					if (use_backup) {
						printf("Non-production device; allowing boot to continue!\n");
					} else {
						err = -3;
						authfail(err);
					}
				}
			}
		}
	}

	if (err == 0) {
		printf("Authenticate image passed!\n");
		return 0;
	}

exit_authfail:
	return err;
}

static bool can_boot(int err, unlock_state_t unlock_state)
{
	if (err) {
		goto no_boot;
	}
	if (has_authenticated == HAS_AUTHENTICATED_KEY) {
		goto ok_to_boot;
	} else if ((has_authenticated == NO_AUTH_KEY) &&
	           (unlock_state == UNLOCK_STATE_NON_PROD)) {
		goto ok_to_boot;
	}
no_boot:
	return false;
ok_to_boot:
	return true;
}

#ifdef CONFIG_CMD_OTPCTRL
#define SECUREBOOT_CMD 0x02
#define AR_CMD         0x12
#define IPL_CUST_CMD   0x14
#define KERNEL_CMD     0x16
#define IPL_CUST_ANTIBACK_OFFSET       (15)
#define KERNEL_ANTIBACK_OFFSET       (0x24)
#define IPL_CUST_KERNEL_REDUNDANT_BYTES (2)
#define IPL_CUST_KERNEL_MAX_VER        (24)
#define IPL_CUST_KERNEL_BIT_OR          (2)
#define OTPWRITEBITS                   (32)
#define E_LOADER_CUST                   (0)
#define E_LOADER_KERNEL                 (1)
extern void otp_ctrl_W(unsigned long  otp_cmd,unsigned long  val);
extern void otp_ctrl_R(unsigned long  otp_cmd);
extern void otp_show_results(unsigned long op, u8* dataArr, unsigned long dataSize);
extern void chip_flush_miu_pipe(void);
static u8 ar_val;
static u64 ipl_cust_kernel_anti_val;
#define ASCII_TO_NUM(asic)     ({       \
    char num=0;                         \
    if((asic)>='A' && (asic)<='F')      \
    {                                   \
        num = asic-55;                  \
    }                                   \
    else if((asic)>='0' && (asic)<='9') \
    {                                   \
        num = asic-'0';                 \
    }                                   \
    else                                \
    {                                   \
        num = 'F';                      \
    }                                   \
    num;                                \
})
static void get_option(u8 *u8antiv, void *buf, int enType)
{
	chip_flush_miu_pipe();
	u8 Version1=0,Version2=0;
	switch(enType)
	{
		case E_LOADER_CUST:
			*u8antiv = (*(volatile char *)((u32)buf+IPL_CUST_ANTIBACK_OFFSET));
			break;
		case E_LOADER_KERNEL:
			Version1 = (*(volatile char *)((U32)buf+KERNEL_ANTIBACK_OFFSET));
			Version2 = (*(volatile char *)((U32)buf+KERNEL_ANTIBACK_OFFSET+1));
			*u8antiv = (ASCII_TO_NUM(Version1) << 4) | ASCII_TO_NUM(Version2);
			break;
		default:
			break;
	}
}
static void get_otp_version(u8 *u8anti_cnt, int enType)
{
	unsigned long op_r = 0x2;
	unsigned long otp_cmd = (enType == E_LOADER_KERNEL) ? KERNEL_CMD : IPL_CUST_CMD;
	unsigned long dataSize;
	u64 AntiV_BitsORValue = 3;
	u64 AntiV_BitsOR = 2;
	u8 cnt = 0;

	ipl_cust_kernel_anti_val = 0;
	dataSize = sizeof(ipl_cust_kernel_anti_val)-IPL_CUST_KERNEL_REDUNDANT_BYTES;
	otp_ctrl_R((otp_cmd << CMD_OFFSET) | 0x00);
	otp_show_results(op_r, (u8*)&ipl_cust_kernel_anti_val, dataSize);
	while (ipl_cust_kernel_anti_val) {
		if (ipl_cust_kernel_anti_val & AntiV_BitsORValue) {
			cnt += 1;
		}
		ipl_cust_kernel_anti_val = (ipl_cust_kernel_anti_val >> AntiV_BitsOR);
	}
	*u8anti_cnt = cnt;
}
static void update_otp_version(u8 u8antiv, int enType)
{
	unsigned long op_w = 0x1;
	unsigned long otp_cmd = (enType == E_LOADER_KERNEL) ? KERNEL_CMD : IPL_CUST_CMD;
	unsigned long value;
	u8 off = 0;
	u8 OTPV = u8antiv * IPL_CUST_KERNEL_BIT_OR;
	while (OTPV != 0) {
		if(OTPV >= OTPWRITEBITS) {
			value = ((1ULL << OTPWRITEBITS)-1);
			OTPV -= OTPWRITEBITS;
		} else {
			value = (1UL << OTPV)-1;
			OTPV = 0;
		}
		otp_ctrl_W((otp_cmd << CMD_OFFSET)|off, value);
		otp_show_results(op_w, NULL, 0);
		off += 4;
	}
}
#endif
static int boot_recovery_kernel(unlock_state_t unlock_state, uint8_t reboot_reason, bool use_backup)
{
	int err = -1;
	u_char *kernel_load_address = (u_char *)COCOA_RECOVERY_KERNEL_LOAD_ADDRESS;
	static char *argv[1];
	u32 kernel_size;

	set_ring_recovery_command_line(unlock_state, reboot_reason, use_backup);
	err = read_nand_partition(use_backup ? STR(COCOA_RECOVERY_BACKUP_KERNEL_PART_NAME) : STR(COCOA_RECOVERY_KERNEL_PART_NAME),
	                          sizeof(image_header_t),
	                          kernel_load_address, 0);

	if (err) {
		printf("Failed to read kernel header from NAND\n");
		goto exit;
	}

	kernel_size = image_get_image_size((image_header_t *)kernel_load_address);
	if (kernel_size > COCOA_KERNEL_SIZE) {
		debugf("Invalid kernel image size\n");
		err = -2;
		goto exit;
	}

	err = read_nand_partition(use_backup ? STR(COCOA_RECOVERY_BACKUP_KERNEL_PART_NAME) : STR(COCOA_RECOVERY_KERNEL_PART_NAME),
	                          kernel_size + COCOA_KERNEL_SIG_SIZE,
	                          kernel_load_address, 0);

	if (err) {
		printf("Failed to read kernel image from NAND\n");
		goto exit;
	}

	err = authenticate_image((u32)kernel_load_address, kernel_size,
	                         KERN_KEY_OFFSET, unlock_state, use_backup);

	if (can_boot(err, unlock_state)) {
#ifdef CONFIG_CMD_OTPCTRL
		if (ENABLE_ANTIROLLBACK && ar_val) {
			u8 u8antiv = 0, u8anti_cnt = 0;
			int enType = E_LOADER_KERNEL;
			get_option(&u8antiv, (void*)kernel_load_address, enType);
			get_otp_version(&u8anti_cnt, enType);
			if (u8anti_cnt > u8antiv || u8antiv > IPL_CUST_KERNEL_MAX_VER) {
				// HALT
				printf("HALT, ANTI %d %d\n", u8anti_cnt, u8antiv);
				for (;;);
			} else if (u8antiv > u8anti_cnt) {
				update_otp_version(u8antiv, enType);
			}
		}
#endif
		signal_boot_ack_pulse();
		argv[0] = STR(COCOA_RECOVERY_KERNEL_LOAD_ADDRESS);
		/* Perform bootm processing */
		err = do_bootm_states(NULL, 0, 1, argv,
				BOOTM_STATE_START | BOOTM_STATE_FINDOS |
				BOOTM_STATE_FINDOTHER | BOOTM_STATE_LOADOS |
				BOOTM_STATE_OS_PREP | BOOTM_STATE_OS_FAKE_GO |
				BOOTM_STATE_OS_GO,
				&images, 1);
	}

exit:
	printf("Unexpected exit from %s: err=%d has_authenticated=0x%x!\n", __func__, err, has_authenticated);
	return err;
}


__weak void mmu_disable(void)
{
}

__weak void mmu_enable(void)
{
}

static u32 get_next_stage_key_offset(const char* partition_name)
{
	// use uboot key as default
	u32 key_offset = BL_KEY_OFFSET;
	const char rtos_prefix[] = "RTOS_";
	const char ipl_cust_prefix[] = "IPL_CUST";
	if (strncmp(partition_name, rtos_prefix, sizeof(rtos_prefix) - 1) == 0) {
		key_offset = RTOS_KEY_OFFSET;
	} else if (strncmp(partition_name, ipl_cust_prefix, sizeof(ipl_cust_prefix) - 1) == 0) {
		key_offset = IPL_KEY_OFFSET;
	}
	return key_offset;
}
static int boot_next_stage(unlock_state_t unlock_state, bool use_backup)
{
	int err = -1;
	u_char *ns_boot_address = (u_char *)COCOA_NEXT_STAGE_BOOT_ADDRESS;
	u32 ns_size;

	// read 1 page
	err = read_nand_partition(use_backup ? STR(COCOA_NEXT_STAGE_BACKUP_PART_NAME) : STR(COCOA_NEXT_STAGE_PART_NAME),
		get_ns_header_size(), ns_boot_address, 0);
	if (err) {
		printf("Failed to read next stage header from NAND\n");
		goto exit;
	}

	// parse header for size
	ns_size = get_ns_image_size(ns_boot_address);

	// check size
	if (ns_size > COCOA_NEXT_STAGE_PART_SIZE) {
		debugf("Invalid next stage image size\n");
		err = -2;
		goto exit;
	}

	// read with correct size
	err = read_nand_partition(use_backup ? STR(COCOA_NEXT_STAGE_BACKUP_PART_NAME) : STR(COCOA_NEXT_STAGE_PART_NAME),
		ns_size + COCOA_NEXT_STAGE_SIG_SIZE, ns_boot_address, 0);

	if (err) {
		printf("Failed to read next stage image from NAND\n");
		goto exit;
	}

	err = authenticate_image((u32)ns_boot_address, ns_size,
	                         get_next_stage_key_offset(STR(COCOA_NEXT_STAGE_PART_NAME)),
	                         unlock_state, use_backup);
	if (can_boot(err, unlock_state)) {
#ifdef CONFIG_CMD_OTPCTRL
		if (ENABLE_ANTIROLLBACK && ar_val) {
			u8 u8antiv = 0, u8anti_cnt = 0;
			int enType = E_LOADER_CUST;
			get_option(&u8antiv, (void*)ns_boot_address, enType);
			get_otp_version(&u8anti_cnt, enType);
			if (u8anti_cnt > u8antiv || u8antiv > IPL_CUST_KERNEL_MAX_VER) {
				// HALT
				printf("HALT, ANTI %d %d\n", u8anti_cnt, u8antiv);
				for (;;);
			} else if (u8antiv > u8anti_cnt) {
				update_otp_version(u8antiv, enType);
			}
		}
#endif
		/* The nand driver appears not to invalidate cache ranges, so
		   push this cache enable to the last possible moment. */
		dcache_enable();
		icache_enable();
		mmu_enable();
		signal_boot_ack_pulse();
		((void *(*)(void))ns_boot_address)();
		mmu_disable();
		dcache_disable();
		icache_disable();
		/* Continuing here is a (highly unlikely) error case. Allow it
		   to be detected */
		err = -6;
	}

exit:
	printf("Unexpected exit from %s: err=%d has_authenticated=0x%x!\n", __func__, err, has_authenticated);
	return err;
}

extern int unlocker_get_challenge(char *unlock_code, int unlock_code_len);
extern int unlocker_write_signature(uint8_t *unlock_signature, int unlock_sig_len);
extern int unlocker_relock_device(bool reboot);

int get_unlock_code(char *unlock_code, int unlock_code_len)
{
	return unlocker_get_challenge(unlock_code, unlock_code_len);
}

int get_lock_state(char *lock_state, int lock_state_len)
{
	int err = 0;
	unlock_state_t unlock_state;
	is_unlocked(&unlock_state);

	switch (unlock_state) {
		case UNLOCK_STATE_LOCKED:
			strlcpy(lock_state, "UNLOCK_STATE_LOCKED", lock_state_len);
			break;

		case UNLOCK_STATE_RESTRICTED_UNLOCK:
			strlcpy(lock_state, "UNLOCK_STATE_RESTRICTED_UNLOCK", lock_state_len);
			break;

		case UNLOCK_STATE_FULL_UNLOCK:
			strlcpy(lock_state, "UNLOCK_STATE_FULL_UNLOCK", lock_state_len);
			break;

		case UNLOCK_STATE_NON_PROD:
			strlcpy(lock_state, "UNLOCK_STATE_NON_PROD", lock_state_len);
			break;

		case UNLOCK_STATE_INVALID:
		default:
			/* Invalid state - return error */
			printf("Invalid unlock state %d\n", unlock_state);
			err = -1;
	}
	return err;
}

int write_unlock_signature(uint8_t *unlock_signature, int unlock_sig_len)
{
	return unlocker_write_signature(unlock_signature, unlock_sig_len);
}

int relock_device(void)
{
	return unlocker_relock_device(false);
}

/* This function is in cmd_nand.c  */
extern int read_nand_partition(const char *part_name, size_t size_to_read,
                               u_char *addr, int dev);

#ifdef COCOA_INCLUDE_KEYS_IN_IMAGE
int load_ring_public_key(u8* key_load_addr, int key_page, int key_offset, int key_size)
{
	int err = -1;
	u8* src_key_ptr = NULL;
	bool dev = (key_page == DEV_KEY_PAGE);
	switch (key_offset)
	{
		case KERN_KEY_OFFSET:
			src_key_ptr = dev ? kernel_dev_key : kernel_prod_key;
			break;
		case RTOS_KEY_OFFSET:
		case IPL_KEY_OFFSET:
			// Only one of these is required per product by U-Boot - mapping is done elsewhere
			src_key_ptr = dev ? ns_dev_key : ns_prod_key;
			break;
		case FULL_UNLOCK_KEY_OFFSET:
			src_key_ptr = dev_unlock_key;
			break;
		case MFG_UNLOCK_KEY_OFFSET:
			src_key_ptr = vendor_unlock_key;
			break;
		default:
			src_key_ptr = NULL;
			break;
	}
	if (src_key_ptr) {
		memcpy(key_load_addr, src_key_ptr, key_size);
		err = 0;
	}
	return err;
}
#else
int load_ring_public_key(u8* key_load_addr, int key_page, int key_offset, int key_size)
{
	int err = -1;
	static const int load_size = 2048; /* 1 NAND page */
	const uint32_t max_offset_idx = getotpversion() - 1;
	u8 *tmpkey = malloc(load_size);
	if (!tmpkey) {
		printf("Failed to allocate space to load key\n");
		goto out;
	}
	if (key_offset + key_size > load_size) {
		printf("Requesting key outside page area!\n");
		goto out_free;
	}

	err = read_otp_page(key_page, tmpkey, load_size, 0);
	if (err) {
		printf("Failed to read ring public key from NAND OTP\n");
		goto out_free;
	}
	if (key_page == PROD_KEY_PAGE && key_offset > PROD_PAGE_MAX_OFFSETS[max_offset_idx]) {
		goto out_free;
	}
	if (key_page == DEV_KEY_PAGE && key_offset > DEV_PAGE_MAX_OFFSETS[max_offset_idx]) {
		// it is needed because ipl_key is absent in page 1
		printf("tried to load non-existent ring public key from DEV OTP page\n");
		key_offset = 0;
	}
	memcpy(key_load_addr, tmpkey + key_offset, key_size);
out_free:
	free(tmpkey);
out:
	return err;
}
#endif

static long getenv_long(const char *name, int base, long default_val)
{
	const char *str = getenv(name);

	return str ? simple_strtol(str, NULL, base) : default_val;
}

char *getserial(void)
{
	int err = 0;
	char *dsn_p = NULL;

	err = try_otp_read_all(dsn, boardid);

	if (err == ERR_METADATA_READ_OK) {
		dsn_p = dsn;
	} else {
		char *env_dsn_p = getenv("serial#");
		if (env_dsn_p) {
			dsn_p = env_dsn_p;
		} else if (err == ERR_METADATA_USE_DEFAULTS) {
			dsn_p = dsn;
		}
	}
	return dsn_p;
}

static int get_major_minor_revision(enum RB_MAJOR_REVS* major_rev, int* minor_rev, int* second_source_ver, enum RB_PROD_ID_TYPES* product_type)
{
	int err = try_otp_read_all(dsn, boardid);
	if (err >= ERR_METADATA_READ_OK) {
		if (major_rev) {
			if (boardid[BOARD_ID_MAJOR_REV_INDEX] >= RB_MAJOR_REV_MIN &&
				boardid[BOARD_ID_MAJOR_REV_INDEX] < RB_MAJOR_REV_MAX) {
				*major_rev = boardid[BOARD_ID_MAJOR_REV_INDEX];
			}
		}
		if (minor_rev) {
			*minor_rev = boardid[BOARD_ID_MINOR_REV_INDEX] - '0';
		}
		if (second_source_ver) {
			*second_source_ver =
				(boardid[BOARD_ID_2ND_SOURCE_INDEX] - '0') * 10 +
				(boardid[BOARD_ID_2ND_SOURCE_INDEX + 1] - '0');
		}
		if (product_type) {
			for (int i = 0; i < ARRAY_SIZE(BOARD_ID_PROD_ID_STRS); i++) {
				if (strncmp(&boardid[BOARD_ID_PROD_ID_INDEX],
						BOARD_ID_PROD_ID_STRS[i],
						BOARD_ID_PROD_ID_LEN) == 0) {
					*product_type = i;
					break;
				}
			}
		}
	}
	return err;
}

const char *getsensor(void)
{
	const char *sensor = ps5250_str;

	enum RB_PROD_ID_TYPES product_type = RB_PRODUCT_UNKNOWN;
	enum RB_MAJOR_REVS    major_rev = RB_MAJOR_REV_UNKNOWN;
	int second_source_ver;
	int minor_rev;

	int err = get_major_minor_revision(&major_rev, &minor_rev, &second_source_ver, &product_type);
	if (err >= ERR_METADATA_READ_OK) {
		switch (product_type) {
		case RB_PRODUCT_ABC:
			if (second_source_ver > 0)
				sensor = sp2309_str;
			break;
		case RB_PRODUCT_ABG:
			/* ABG is switching to SP2309 from DVT2.1 (DVT3 in internal numbering) */
			if ((major_rev > RB_MAJOR_REV_DVT) ||
			    ((major_rev == RB_MAJOR_REV_DVT) && (minor_rev >= 3)))
				sensor = sp2309_str;
			break;
		case RB_PRODUCT_ABP:
			sensor = sp2309_str;
			break;
		case RB_PRODUCT_ABN:
			sensor = sns4_str;
			break;
		case RB_PRODUCT_ABM:
			sensor = ps5416_str;
			break;
		case RB_PRODUCT_ABO:
			snprintf(sensorstr, SENSOR_STR_LEN, "%s+%s", sns3_str, sns2_str);
			sensor = sensorstr;
			break;
		case RB_PRODUCT_ABS:
			sensor = os04c10_str;
			break;
		case RB_PRODUCT_ABE:
		case RB_PRODUCT_ABF:
			sensor = sns1_str;
			break;
		case RB_PRODUCT_ABH:
			sensor = bt656_str;
			break;
		default:
			break;
		}
	}
	return sensor;
}

/* Erase an mtd partition
 * Returns true if this succeeds
 */
static bool erase_nand(const char *part_name, int dev)
{
	int idx;
	loff_t max_size;
	nand_erase_options_t opts;

	memset(&opts, 0, sizeof(opts));
	if (get_part(part_name, &idx, &opts.offset, &opts.length, &max_size) != 0) {
		printf("erase_nand failed to find part\n");
		return false;
	}

	printf("erase_nand device %d offset 0x%llx, size 0x%llx\n", dev, opts.offset, opts.length);
	nand_info_t *nand = &nand_info[dev];
	int ret = nand_erase_opts(nand, &opts);
	if (ret != 0) {
		printf("erase_nand failed %d\n", ret);
		return false;
	}
	return true;
}

#ifdef CONFIG_CMD_OTPCTRL
#define UNINITIALIZED_VAL       0xff
#define RETRY_MAX                  2
#define SPINAND_SKIP_SD         0x04
#define JTAG_DEBUG_DISABLE_CMD  0x10
#define BOOT_SRC_CMD            0x1c
#define CID_CMD                 0x09
void otp_readback(unsigned long  otp_cmd, u8* data, unsigned long dataSize) {
	const unsigned long op_r = 0x2;
	u8 retry = 0;
	do {
		otp_ctrl_R((otp_cmd << CMD_OFFSET) | 0x00);
		otp_show_results(op_r, data, dataSize);
		if (*data != UNINITIALIZED_VAL) {
			break;
		}
		retry++;
	} while(retry < RETRY_MAX);
}
// For device using SAV533, we would check chip OTP programming status for production lock validation
// below is the 3 key value readback for checking ->
// BOOTSRC
// JTAG DISABLE
// CID
static bool validate_production_state()
{
	u8 jtag_debug_disable_val = UNINITIALIZED_VAL, bootsrc_val = UNINITIALIZED_VAL, cid_val = UNINITIALIZED_VAL;
	unsigned long otp_cmd = JTAG_DEBUG_DISABLE_CMD; // disable jtag
	otp_readback(otp_cmd, &jtag_debug_disable_val, sizeof(jtag_debug_disable_val));
	otp_cmd = BOOT_SRC_CMD; // bootsrc
	otp_readback(otp_cmd, &bootsrc_val, sizeof(bootsrc_val));
	otp_cmd = CID_CMD;
	otp_readback(otp_cmd, &cid_val, sizeof(cid_val));
	if (jtag_debug_disable_val == UNINITIALIZED_VAL || bootsrc_val == UNINITIALIZED_VAL || cid_val == UNINITIALIZED_VAL) {
		return false;
	}
	if (!jtag_debug_disable_val || bootsrc_val != SPINAND_SKIP_SD || cid_val != CID) {
		return false;
	}
	return true;
}

static void do_check_valid_otp(void)
{
	u8 secureboot_val = UNINITIALIZED_VAL;
	ar_val = UNINITIALIZED_VAL;
	unsigned long otp_cmd = SECUREBOOT_CMD; // secure boot
	enum RB_PROD_ID_TYPES product_type = RB_PRODUCT_UNKNOWN;
	enum RB_MAJOR_REVS    major_rev = RB_MAJOR_REV_UNKNOWN;
	int second_source_ver;
	int minor_rev;
	int status = get_major_minor_revision(&major_rev, &minor_rev, &second_source_ver, &product_type);

	otp_readback(otp_cmd, &secureboot_val, sizeof(secureboot_val));
	otp_cmd = AR_CMD; // antirollback
	otp_readback(otp_cmd, &ar_val, sizeof(ar_val));

	if(ENABLE_ANTIROLLBACK) {
		// only halt when one is enabled and another is not.
		if ((secureboot_val == UNINITIALIZED_VAL) || (ar_val == UNINITIALIZED_VAL) || (secureboot_val ^ ar_val)) {
			if ((status >= ERR_METADATA_READ_OK) &&
			    (product_type == RB_PRODUCT_ABP || product_type == RB_PRODUCT_ABN) &&
			    ((major_rev < RB_MAJOR_REV_DVT) ||
			     ((major_rev == RB_MAJOR_REV_DVT) && (minor_rev < 2)))) {
				/* Don't halt for early-revision ABP & ABN devices.
				   Alpha (EVT1.2) and Beta1 (DVT1) trials devices have secure
				   boot but not anti-rollback enabled in hardware */
			} else {
				printf("HALT secureboot_val != ar_val\n");
				for (;;);
			}
		}
	} else {
		// only halt when AR and SECUREBOOT is both enabled in Hardware and not in SW.
		if (secureboot_val && ar_val) {
			printf("HALT secureboot_val == ar_val = 1\n");
			for (;;);
		}
	}
	if (secureboot_val) {
		if (status < ERR_METADATA_READ_OK) {
			printf("HALT failed to read boardid");
			for(;;);
		} else if (major_rev == RB_MAJOR_REV_PVT || major_rev == RB_MAJOR_REV_MP) {
			if (false == validate_production_state()) {
				printf("HALT invalid prod state\n");
				for (;;);
			}
		} else if (major_rev < RB_MAJOR_REV_MIN || major_rev >= RB_MAJOR_REV_MAX) {
			printf("HALT invalid boardid");
			for(;;);
		}
	}
}
#endif

void do_factory_reset(ring_config *config)
{
	if (config == NULL)
		return;

	/* Clear any unlock codes before we try to boot further */
	relock_device();

	/* Wipe env partition (use default env) */
	erase_nand(COCOA_ENV_NAME, 0);
	set_default_env("## Factory reset - revert to default environment\n");

	/* Reset ring config to default state */
	ring_config_reset(config);
	set_ring_config(config);
}

/**
 * Perform validation of UBI partitions for factory purposes
 *
 * This function only validates the presence of the partitions;
 * there is separate code to handle attempting to recover corrupted
 * or missing partitions.  Note that we can't validate the customer
 * partition is mountable, since that won't always be the case here.
 * It doesn't seem to be possible to create a mountable ubifs partition
 * from within U-Boot - only create the volume and check an already
 * created partition is mountable.  Hence this function does no more
 * than basic validation that the expected partitions are present.
 */
int validate_ubi_partitions(void)
{
	int err = 0;

	err = ubi_part(COCOA_UBI_NAME, NULL);
	if (err != 0) {
		printf("Failed: reading UBI\n");
		return err;
	}

	err = ubi_check(COCOA_RECOVERY_PARTITION);
	if (err != 0) {
		printf("Failed: recovery partition check\n");
		return err;
	}

	err = ubi_check_size_enough(COCOA_RECOVERY_PARTITION, COCOA_RECOVERY_SIZE);
	if (err != 0) {
		printf("Failed: recovery partition size check\n");
		return err;
	}

	err = ubi_check(STR(COCOA_DATA_PARTITION));
	if (err != 0) {
		printf("Failed: data partition check\n");
		return err;
	}

	err = ubi_check(COCOA_BOOT_CONFIG_PARTITION);
	if (err != 0) {
		printf("Failed: ring config check\n");
		return err;
	}
	return 0;
}

void do_ubifs_management(bool factory_reset)
{
	unsigned long before_ubi = get_timer(0);
	int err = 0;

	ubifs_init();

	/* Recover/recreate recovery partition */
	err = ubi_check_size_enough(COCOA_RECOVERY_PARTITION, COCOA_RECOVERY_SIZE);
	if (0 != err) {
		// Not big enough, remove before recreating
		if (-EFBIG == err) {
			ubi_remove_vol(COCOA_RECOVERY_PARTITION);
		}

		printf("Recreating %s partition\n", COCOA_RECOVERY_PARTITION);

		// Try creating the partition, the update process should rewrite the whole volume
		err = ubi_create_vol(COCOA_RECOVERY_PARTITION, COCOA_RECOVERY_SIZE, 1);
		if (-ENOSPC == err) {
			printf("Recreate %s failed, remove %s and retry\n", COCOA_RECOVERY_PARTITION, STR(COCOA_DATA_PARTITION));
			ubi_remove_vol(STR(COCOA_DATA_PARTITION));
			err = ubi_create_vol(COCOA_RECOVERY_PARTITION, COCOA_RECOVERY_SIZE, 1);
		}
		printf("Recreate %s %s err=%d\n", COCOA_RECOVERY_PARTITION, err == 0 ? "success" : "failed", err);
	}

	/* Recover/recreate customer partition */
	int userdata_corrupt = (uboot_ubifs_mount("ubi:" STR(COCOA_DATA_PARTITION)) != 0);
	if (userdata_corrupt || factory_reset) {
		printf("Recreating %s partition corrupt=%d factory_reset=%d\n", STR(COCOA_DATA_PARTITION), userdata_corrupt, factory_reset);
		ubi_remove_vol(STR(COCOA_DATA_PARTITION));
		// We can give over all the rest of the partition to userdata
		ubi_create_max_size_vol(STR(COCOA_DATA_PARTITION), 1);
	}
	printf("%lums to manage ubifs\n", get_timer(before_ubi));
}

void do_device_boot(ring_config *config, unlock_state_t unlock_state)
{
	if (config == NULL)
		return;

	// Check if normal boot is getting far enough to clear the attempt count.  This implies
	// that ubifs has mounted - by that point we should be in a good enough position to
	// OTA update via the normal mechanism so recovery OTA is not required.
	if (config->boot.normalBootAttemptCount >= MAX_NORMAL_MODE_BOOT_COUNT &&
	    config->boot.recoveryMode == 0) {
		printf("NORMAL BOOT EXCEEDED FAIL COUNT - REQUIRES RECOVERY\n");
		config->boot.recoveryMode = 1;
		config->boot.rebootReason = REBOOT_REASON_RECOVERY_STG1_BOOT;
	}

	if (config->boot.recoveryMode) {
		// Keep a count of recovery boots
		config->boot.recoveryBootCount++;
		config->boot.normalBootAttemptCount = 0;

		printf("Recovery Boot count %d/%d\n", config->boot.recoveryBootCount, MAX_RECOVERY_MODE_BOOT_COUNT);

		if (config->boot.recoveryBootCount <= MAX_RECOVERY_MODE_BOOT_COUNT) {
			set_ring_config(config);
			if ((config->boot.recoveryMode == 1) &&
			    (config->boot.rebootReason == REBOOT_REASON_RECOVERY_STG2_BOOT) &&
			    (check_shadow_recovery() == false)) {
				printf("Shadow recovery partition check failed! Needs retry\n");
				/* Stage 2 boot needs valid shadow partition.  We will have to try
					*  again to create it */
				config->boot.rebootReason = REBOOT_REASON_RECOVERY_STG1_BOOT;
				set_ring_config(config);
			}

			if (config->boot.rebootReason == REBOOT_REASON_RECOVERY_STG2_BOOT || check_recovery_mountable()) {
				boot_recovery_kernel(unlock_state, config->boot.rebootReason, false); // no return on success
				printf("Trying backup recovery kernel\n");
				boot_recovery_kernel(unlock_state, config->boot.rebootReason, true); // try the backup kernel
			}
		}

		/* If the recovery mode kernel is unbootable,
		 * or we've tried to boot too many times into recovery
		 * clear state which forces boot into that mode */
		config->boot.recoveryMode = 0;
		config->boot.rebootReason = (uint8_t)((config->boot.rebootReason == REBOOT_REASON_APPLY_OTA) ?
						       REBOOT_REASON_FAILED_OTA :
						       REBOOT_REASON_NORMAL);
	} else {
		config->boot.recoveryBootCount = 0;
		config->boot.normalBootAttemptCount++;
		printf("Boot attempt count: %d\n", config->boot.normalBootAttemptCount);
	}
	set_ring_config(config);

	boot_next_stage(unlock_state, false); // no return on success
	printf("Trying backup\n");
	boot_next_stage(unlock_state, true); // Try the backup image

	// this function is only expected to return in the case that all signature validation attempts failed
}

/* Normal boot;
 * nand read.e COCOA_RTOS_BOOT_ADDRESS COCOA_RTOS_PART_NAME COCOA_RTOS_SIZE
 * go COCOA_RTOS_BOOT_ADDRESS
 *
 * nand read.e COCOA_RECOVERY_KERNEL_LOAD_ADDRESS COCOA_RECOVERY_KERNEL_PART_NAME COCOA_KERNEL_SIZE
 * bootm COCOA_RECOVERY_KERNEL_LOAD_ADDRESS
 */
int do_ring_autoboot(void)
{
	unlock_state_t unlock_state;
	ring_config config;

	/* The NAND driver doesn't flush / invalidated cache ranges.  The code below
	 * assumes the caches are disabled (they are on SAV526 in IPL).  Make SAV533
	 * behaviour match by disabling the caches here.  They are enabled before
	 * booting to the next stage. */
	dcache_disable();
	icache_disable();

	configure_boot_ack();
	signal_boot_ack_pulse();
#ifdef CONFIG_AUTO_TRY_FASTBOOT
#ifndef CONFIG_AUTO_TRY_FASTBOOT_TIMEOUT_MS
#define CONFIG_AUTO_TRY_FASTBOOT_TIMEOUT_MS 2000
#endif

	/* Negative value will skip fastboot. 0 will remain in fastboot */
	static long fastbootwaitms = -1;

	if (fastbootwaitms < 0)
		fastbootwaitms = getenv_long("fastbootwaitms", 10,
		                             CONFIG_AUTO_TRY_FASTBOOT_TIMEOUT_MS);

	if (fastbootwaitms >= 0)
		fastboot_try(fastbootwaitms);
#endif

	is_unlocked(&unlock_state);

	/* Locked devices don't get access to the command line */
	if ((unlock_state > UNLOCK_STATE_LOCKED) && had_ctrlc()) {
		cli_init();
		cli_loop();
	}

	// Load ubi partition
	if (ubi_part(COCOA_UBI_NAME, NULL) != 0) {
		printf("failed to read UBI\n");

		// This should recreate ubi in a good state
		erase_nand(COCOA_UBI_NAME, 0);
		ubi_part(COCOA_UBI_NAME, NULL);
	}

	// This will recreate bootconfig if needed
	get_ring_config(&config);

	if (unlock_state == UNLOCK_STATE_RESTRICTED_UNLOCK) {
		platform_increment_boot_count();
		/* Re-evaluate unlock state */
		is_unlocked(&unlock_state);
	}

	int factory_reset = (config.boot.rebootReason == REBOOT_REASON_FACTORY_RESET);
	if (factory_reset) {
		do_factory_reset(&config);
		/* Re-evaluate unlock state */
		is_unlocked(&unlock_state);
	}
	do_ubifs_management(factory_reset);

#ifdef CONFIG_CMD_OTPCTRL
	do_check_valid_otp();
#endif

	do_device_boot(&config, unlock_state);
	// boot has failed if we get here

	if (unlock_state < UNLOCK_STATE_NON_PROD) {
		/* Production device; halt on boot error */
		printf("HALT\n");
		for (;;) ;
	}

#ifdef CONFIG_AUTO_TRY_FASTBOOT
	/* We failed to boot.  On retry just stay in fastboot */
	fastbootwaitms = 0;
	clear_ctrlc();
#endif

	return 0;
}

char *getboardid(void)
{
	int err = try_otp_read_all(dsn, boardid);
	char *boardid_p = NULL;

	if (err >= ERR_METADATA_READ_OK) {
		boardid_p = boardid;
	}
	return boardid_p;
}

uint32_t getotpversion(void)
{
	uint32_t version;
	int err = try_otp_read_version(&version);
	if (err < ERR_METADATA_READ_OK) {
		version = 1;
	}
	return version;
}
